local inv = require("modules.inv");
local config = require("../../config") ---@type Config
local hmac = require("lib.hmac-pbkdf2-aead")
---@class RemoteAccess
---@field type string
---@field data RemoteAccessData

---@class RemoteAccessData
---@field itemName string|nil
---@field amount number|nil
---@field slots number[]|nil

local function run()
  if config.remote == nil then return end

  local modem = peripheral.wrap(config.remote.modem)
  if not modem then return end

  for _, remote in pairs(config.remote.remotes) do
    if not modem.isOpen(remote.port) then
      modem.open(remote.port)
    end
  end

  ---@type ccTweaked.peripheral.Inventory
  local ender_storage = peripheral.wrap(config.remote.ender_storage)

  while true do
    local _, _, channel, _, message, _ = os.pullEvent("modem_message")

    for color, remote in pairs(config.remote.remotes) do
      if remote.port == channel then
        local err, data = pcall(function()
          return hmac.decrypt_with_password(remote.password, message)
        end)

        if not data then return end

        if err == true then
          ---@type RemoteAccess
          local json, errj = textutils.unserialiseJSON(data)
          if json or errj ~= nil then
            local parts = {}
            for part in string.gmatch(color, "[^/]+") do
              table.insert(parts, part)
            end
            if #parts ~= 3 then error("Invalid frequency string (need 3 colors)") end
            ender_storage.setFrequency(
              colors[parts[1]],
              colors[parts[2]],
              colors[parts[3]]
            )
            if json.type == "withdraw" then
              inv.sendItemToSelf(json.data.itemName, config.remote.ender_storage, json.data.amount)
            elseif json.type == "deposit" then
              if json.data.itemName then
                local move = {}
                for s, i in pairs(ender_storage.list()) do
                  if i.name == json.data.itemName then
                    table.insert(move, s)
                  end
                end
                inv.sendItemAwayMultiple(move, ender_storage, config.remote.ender_storage, json.data.amount)
              elseif json.data.slots then
                if type(json.data.slots) ~= 'table' then
                  return
                end

                inv.sendItemAwayMultiple(json.data.slots, ender_storage, config.remote.ender_storage, json.data.amount)
              end
            end
          end
        end
      end
    end
  end
end

return {
  run = run
}
