--[[
main goals in this application:
 - make UI for taking items out of a list of approved
]]

local pprint = require('cc.pretty').pretty_print

---@class Config
---@field inventories string[]
---@field import string[]|nil
---@field hard_limit number|nil

local config = require("../config") ---@type Config

local abstractInventoryLib = require("abstractInventoryLib")

local peripherals = peripheral.getNames()

-- DO NOT MODIFY THE INVENTORIES EXTERNALLY,
--   or.. if you do, call the resync functions

---@type table<string, number>
local itemsLUT = {}
local HARD_LIMIT = config.hard_limit
local turtleMoveAllowed = true
---@type AbstractInventory
local ail = nil
local turtleId = peripheral.find("modem").getNameLocal()

if HARD_LIMIT == nil then
    HARD_LIMIT = math.huge
end

function sync()
    print("Syncing..")
    ---@type string[]
    local foundInventories = {}

    for _, invPattern in ipairs(config.inventories) do
        for _, inv in ipairs(peripherals) do
            if string.find(inv, invPattern) then
                table.insert(foundInventories, inv)
            end
        end
    end
    print("total inventories: " .. #foundInventories)
    local total = 0

    for _, inv in ipairs(foundInventories) do
        local perip = peripheral.wrap(inv)
        if perip == nil then
            print("While creating itemLUT, peripheral dissapeared. Please rerun The Storage Solution.")
            return
        end
        for slot, item in pairs(perip.list()) do
            if itemsLUT[item.name] then
                itemsLUT[item.name] = itemsLUT[item.name] + item.count
            else
                itemsLUT[item.name] = item.count
            end

            total = total + item.count

            if total > HARD_LIMIT then break end
        end
        if total > HARD_LIMIT then break end
    end

    ail = abstractInventoryLib(foundInventories, true)

    print("Synced. Total item count: " .. total)
end

---@param itemName string
local function sendItemToSelf(itemName)
    turtleMoveAllowed = false
    if itemsLUT[itemName] then
        local ok, amount = pcall(function()
            ail.performTransfer()

            -- Get all NBT variants for this item
            local nbtList = ail.listNBT(itemName)
            local chosenNBT = nil
            if nbtList and #nbtList > 0 then
                -- Pick a random NBT entry
                chosenNBT = nbtList[math.random(1, #nbtList)]
            end

            local amount = ail.pushItems(turtleId, itemName, 64, nil, chosenNBT, {
                ["allowBadTransfers"] = true,
                ["optimal"] = true
            })

            ail.performTransfer()

            return amount
        end)
        itemsLUT[itemName] = itemsLUT[itemName] - amount
        if itemsLUT[itemName] <= 0 then
            itemsLUT[itemName] = nil
        end
    end
    sleep(0.5)
    turtleMoveAllowed = true
end


---@param slots ccTweaked.turtle.turtleSlot[]
---@param perip ccTweaked.peripheral.Inventory|nil
---@param id string|nil
local function sendItemAwayMultiple(slots, perip, id)

    if perip == nil or id == nil then
        perip = turtle
        id = turtleId
    end

    local itemsInSlots = {}
    for _, slot in ipairs(slots) do
        local item = perip.getItemDetail(slot)
        if item then
            itemsInSlots[slot] = item
        end
    end

    local amount = 0

    for _, slot in ipairs(slots) do
        local ok, pulled = pcall(function()
            ail.performTransfer()
            local pulled =  ail.pullItems(id, slot, 64, nil, nil, {
                ["allowBadTransfers"] = true,
                ["optimal"] = true
            })
            ail.performTransfer()
            return pulled
        end)
        if ok and pulled then
            amount = amount + pulled
        end
    end

    local numItems = 0
    for _, _ in pairs(itemsInSlots) do
        numItems = numItems + 1
    end

    if numItems > 0 then
        local perItemAmount = math.floor(amount / numItems)
        for _, item in pairs(itemsInSlots) do
            if not itemsLUT[item.name] then
                itemsLUT[item.name] = perItemAmount
            else
                itemsLUT[item.name] = itemsLUT[item.name] + perItemAmount
            end
        end

        os.queueEvent("update_ui")
    end
end

local function worstUIEver()
    local term = term
    local w, h = term.getSize()
    local state = {
        search = "",
        selected = 1,
        scroll = 0,
        inSearch = true,
    }

    -- UI definition table
    local ui = {
        {type="label", text="Search: ", x=2, y=2, color=colors.white},
        {type="input", bind="search", x=10, y=2, width=w-12, active=function() return state.inSearch end},
        {type="list", bind="items", x=2, y=4, width=w-2, height=h-6, selected="selected", scroll="scroll"},
        {type="label", text="(Enter to take, Tab to search)", x=2, y=h-1, color=colors.gray},
        {type="label", text="rev. " .. fs.open("storage-solution/version", "r").readAll(), x=2, y=h, color=colors.gray},

    }

    local function getFiltered()
        local filtered = {}
        for name, count in pairs(itemsLUT) do
            if state.search == "" or string.find(name:lower(), state.search:lower(), 1, true) then
                table.insert(filtered, {name = name, count = count})
            end
        end
        table.sort(filtered, function(a, b) return a.count > b.count end)
        return filtered
    end

    local function renderUI()
        term.clear()
        for _, elem in ipairs(ui) do
            if elem.type == "label" then
                term.setCursorPos(elem.x, elem.y)
                term.setTextColor(elem.color or colors.white)
                term.write(elem.text)
                term.setTextColor(colors.white)
            elseif elem.type == "input" then
                term.setCursorPos(elem.x, elem.y)
                if (elem.active and elem.active()) then
                    term.setTextColor(colors.yellow)
                else
                    term.setTextColor(colors.white)
                end
                local val = state[elem.bind] or ""
                term.write(val:sub(1, elem.width))
                term.setTextColor(colors.white)
            elseif elem.type == "list" then
                local items = getFiltered()
                local listHeight = elem.height
                local scroll = state[elem.scroll]
                local selected = state[elem.selected]
                -- Clamp scroll
                if scroll > math.max(0, #items - listHeight) then scroll = math.max(0, #items - listHeight) end
                if scroll < 0 then scroll = 0 end
                state.scroll = scroll
                -- Clamp selected
                if selected < 1 then selected = 1 end
                if selected > listHeight then selected = listHeight end
                if selected > #items - scroll then selected = math.max(1, #items - scroll) end
                state.selected = selected
                -- Draw items
                for i = 1, listHeight do
                    local idx = i + scroll
                    local item = items[idx]
                    term.setCursorPos(elem.x, elem.y + i - 1)
                    if item then
                        if not state.inSearch and selected == i then
                            term.setBackgroundColor(colors.gray)
                            term.setTextColor(colors.black)
                        else
                            term.setBackgroundColor(colors.black)
                            term.setTextColor(colors.white)
                        end
                        local nameStr = item.name:sub(1, elem.width - 8)
                        local countStr = tostring(item.count)
                        local line = string.format("%-"..(elem.width-8).."s %5s", nameStr, countStr)
                        term.write(line)
                        term.setBackgroundColor(colors.black)
                        term.setTextColor(colors.white)
                    else
                        term.write(string.rep(" ", elem.width))
                    end
                end
            end
        end
    end

    renderUI()
    while true do
        local e, p1 = os.pullEvent()
        if e == "update_ui" then
            renderUI()
        elseif e == "key" then
            if state.inSearch then
                if p1 == keys.enter then
                    state.inSearch = false
                elseif p1 == keys.backspace then
                    state.search = state.search:sub(1, -2)
                elseif p1 == keys.down then
                    state.inSearch = false
                elseif p1 >= 32 and p1 <= 126 then
                    state.search = state.search .. string.char(p1)
                end
                state.scroll = 0
                state.selected = 1
            else
                local items = getFiltered()
                local listHeight = ui[3].height
                if p1 == keys.up then
                    if state.selected > 1 then
                        state.selected = state.selected - 1
                    elseif state.scroll > 0 then
                        state.scroll = state.scroll - 1
                    end
                elseif p1 == keys.down then
                    if state.selected < listHeight and (state.selected + state.scroll) < #items then
                        state.selected = state.selected + 1
                    elseif (state.scroll + listHeight) < #items then
                        state.scroll = state.scroll + 1
                    end
                elseif p1 == keys.pageUp then
                    local jump = math.min(state.scroll, listHeight)
                    state.scroll = state.scroll - jump
                elseif p1 == keys.pageDown then
                    local jump = math.min(#items - (state.scroll + listHeight), listHeight)
                    state.scroll = state.scroll + jump
                elseif p1 == keys.home then
                    state.scroll = 0
                    state.selected = 1
                elseif p1 == keys["end"] then
                    state.scroll = math.max(0, #items - listHeight)
                    state.selected = math.min(listHeight, #items - state.scroll)
                elseif p1 == keys.enter then
                    local item = items[state.selected + state.scroll]
                    if item then
                        sendItemToSelf(item.name)
                    end
                elseif p1 == keys.tab then
                    state.inSearch = true
                end
            end
            renderUI()
        elseif e == "mouse_click" then
            local _, x, y = os.pullEventRaw("mouse_click")
            -- Search bar click
            if y == ui[2].y and x >= ui[2].x and x <= (ui[2].x + ui[2].width) then
                state.inSearch = true
                renderUI()
            -- List click
            elseif y >= ui[3].y and y < ui[3].y + ui[3].height then
                local idx = y - ui[3].y + 1
                local items = getFiltered()
                if idx >= 1 and idx <= ui[3].height and (idx + state.scroll) <= #items then
                    state.selected = idx
                    state.inSearch = false
                    local item = items[idx + state.scroll]
                    if item then
                        sendItemToSelf(item.name)
                    end
                    renderUI()
                end
            end
        elseif e == "mouse_scroll" then
            local _, dir = os.pullEventRaw("mouse_scroll")
            local items = getFiltered()
            local listHeight = ui[3].height
            if dir == 1 then -- scroll down
                if (state.scroll + listHeight) < #items then
                    state.scroll = state.scroll + 1
                end
            elseif dir == -1 then -- scroll up
                if state.scroll > 0 then
                    state.scroll = state.scroll - 1
                end
            end
            renderUI()
        end
    end
end

local previousInventory = {}

local function getTurtleInventory()
    local inventory = {}
    for slot = 1, 16 do
        local item = turtle.getItemDetail(slot, false)
        inventory[slot] = item
    end
    return inventory
end

local function detectPlayerInsert()
    previousInventory = getTurtleInventory()

    while true do
        os.pullEvent("turtle_inventory")

        local currentInventory = getTurtleInventory()

        if turtleMoveAllowed then
            local newlyAddedSlots = {}

            for slot = 1, 16 do
                local prev = previousInventory[slot]
                local curr = currentInventory[slot]

                if not prev and curr then
                    table.insert(newlyAddedSlots, slot)
                end
            end

            if #newlyAddedSlots > 0 then
                sendItemAwayMultiple(newlyAddedSlots)
            end
        end

        previousInventory = currentInventory
    end
end

local function importMechanism()
    if config.import == nil then
        return
    end
    if #config.import == 0 then
        return
    end

    while true do
        for _, import_from_inv in ipairs(config.import) do
            ---@type ccTweaked.peripheral.Inventory
            local perip = peripheral.wrap(import_from_inv)
            if perip and perip.size then
                local slotsToSend = {}
                for slot = 1, perip.size() do
                    local item = perip.getItemDetail(slot)
                    if item then
                        table.insert(slotsToSend, slot)
                    end
                end
                if #slotsToSend > 0 then
                    sendItemAwayMultiple(slotsToSend, perip, import_from_inv)
                end
            end
        end
        sleep(1)
    end
end
sync()
parallel.waitForAll(ail.run, importMechanism, worstUIEver, detectPlayerInsert, function ()
        -- this function deals with
end)
