---@class InventoryLayerAIL : InventoryBase
local Base = require("modules.inventory.base")

---@class InventoryLayerAIL : InventoryBase
local AIL = setmetatable({}, Base)
AIL.__index = AIL

local config = require("../../config")

---@type fun(list:string[], boolean, table): AbstractInventory
local abstractInventoryLib = require("lib.abstractInventoryLib")

---@type ccTweaked.peripheral.WiredModem
local modem = peripheral.find("modem")

local turtleId = modem and modem.getNameLocal() or "self"

---@return InventoryLayerAIL
function AIL:new()
  ---@type InventoryLayerAIL
  local o = Base.new(self)

  ---@type AbstractInventory|nil
  o.ail = nil

  return o
end

function AIL:sync()
  print("Syncing AIL...")

  ---@type string[]
  local found = {}

  local peripherals = peripheral.getNames()

  for _, pattern in ipairs(config.inventories) do
    for _, name in ipairs(peripherals) do
      if string.find(name, pattern) then
        table.insert(found, name)
      end
    end
  end

  self.ail = abstractInventoryLib(found, true, {})

  print("Defragging..")
  self.ail.defrag()
  print('Defragged.')
  print("AIL synced:", #found, "inventories")
end

---@return fun()
function AIL:run()
  if not self.ail then
    return function() end
  end

  return self.ail.run
end

---@param itemName string
---@param perip ccTweaked.peripheral.Inventory|string|nil
---@param maxAmount integer|nil
---@param id string|nil
---@return integer
function AIL:sendItemToSelf(itemName, perip, maxAmount, id)
  if not self.ail then return 0 end

  local target = perip or id or turtleId
  local remaining = maxAmount or 64
  local total = 0

  return self:_withMoveLock(function()
    if remaining <= 0 then
      return 0
    end

    local nbtList = self.ail.listNBT(itemName)
    if not nbtList or #nbtList == 0 then
      return 0
    end

    ---@type string|nil
    local chosen = nbtList[math.random(1, #nbtList)]
    if chosen == "NONE" then chosen = nil end

    while remaining > 0 do
      local toSend = math.min(64, remaining)

      self.ail.performTransfer()

      local moved = self.ail.pushItems(
        target,
        itemName,
        toSend,
        nil,
        chosen,
        {
          allowBadTransfers = true,
          optimal = false
        }
      ) or 0

      self.ail.performTransfer()

      if moved <= 0 then break end

      remaining = remaining - moved
      total = total + moved
    end

    return total
  end) or 0
end

---@param slots TurtleSlot[]
---@param perip ccTweaked.peripheral.Inventory|nil
---@param id string|nil
---@param maxAmount integer|nil
---@return integer
function AIL:sendItemAwayMultiple(slots, perip, id, maxAmount)
  if not self.ail then return 0 end

  local src = perip or id or turtleId
  local remaining = maxAmount or math.huge
  local total = 0

  for _, slot in ipairs(slots) do
    if remaining <= 0 then break end

    self.ail.performTransfer()

    local moved = self.ail.pullItems(
      src,
      slot,
      math.min(64, remaining),
      nil,
      nil,
      {
        allowBadTransfers = true,
        optimal = false
      }
    )
    self.ail.performTransfer()

    if moved > 0 then
      remaining = remaining - moved
      total = total + moved
    end
  end

  if total > 0 then
    os.queueEvent("update_ui")
  end

  return total
end

---@return string[]
function AIL:listNames()
  return self.ail and self.ail.listNames() or {}
end

---@return table<string, integer>
function AIL:listItemAmounts()
  return self.ail and self.ail.listItemAmounts() or {}
end

---@param name string
function AIL:getItem(name)
  return self.ail and self.ail.getItem(name) or {}
end

return function()
  return AIL:new()
end
