local config = require("../../config") ---@type Config
local inv = require("modules.inv")


local encryption = require("lib.encryption")
---@class RemoteAccess
---@field type string
---@field data RemoteAccessData
---@field id string

---@class RemoteAccessData
---@field itemName string|nil
---@field amount number|nil
---@field slots number[]|nil
---@field manipulator string|nil
---@field index string|nil

local function sendResponse(modem, port, hashedPassword, id, ok, data, err)
  modem.transmit(
    port,
    port,
    encryption.encrypt(hashedPassword,
      textutils.serialiseJSON({
        type = "response",
        id = id,
        ok = ok,
        data = data,
        error = err
      })
    )
  )
end


local function run()
  if config.remote == nil or config.remote.remotes == nil then return end

  local modem = peripheral.wrap(config.remote.modem)
  if not modem then return end

  local remotes = config.remote.remotes or { config.remote.connection }

  for _, remote in pairs(remotes) do
    if not modem.isOpen(remote.port) then
      modem.open(remote.port)
    end
  end

  ---@type EnderStoragePeripheral
  local ender_storage = peripheral.wrap(config.remote.ender_storage) --[[@as EnderStoragePeripheral]]

  while true do
    local _, _, channel, _, message, _ = os.pullEvent("modem_message")

    for color, remote in pairs(remotes) do
      if remote.port == channel then
        local hashed = encryption.hashKey(remote.password)
        local errd, data = pcall(function()
          return encryption.decrypt(hashed, message)
        end)

        if not data then return end

        if errd == true then
          ---@type RemoteAccess
          local json, errj = textutils.unserialiseJSON(data)
          if json or errj ~= nil then
            local parts = {}
            for part in string.gmatch(color, "[^/]+") do
              table.insert(parts, part)
            end
            if #parts ~= 3 then error("Invalid frequency string (need 3 colors)") end
            ender_storage.setFrequency(
              colors[parts[1]],
              colors[parts[2]],
              colors[parts[3]]
            )
            if json.type == "withdraw" then
              local ok, err = pcall(function()
                inv:sendItemToSelf(
                  json.data.itemName,
                  config.remote.ender_storage,
                  json.data.amount
                )
              end)

              sendResponse(modem, remote.port, hashed, json.id, ok, nil, err)
            elseif json.type == "withdraw_to_manipulator" then
              local ok, err = pcall(function()
                inv:sendItemToSelf(json.data.itemName, peripheral.wrap(json.data.manipulator).getInventory(),
                  json.data.amount,
                  json.data.manipulator)
              end)

              sendResponse(modem, remote.port, hashed, json.id, ok, nil, err)
            elseif json.type == "deposit_from_manipulator" then
              local ok, err = pcall(function()
                local manip = peripheral.wrap(json.data.manipulator)
                local move = {}
                for s, i in pairs(manip.getInventory().list()) do
                  if i.name == json.data.itemName then
                    table.insert(move, s)
                  end
                end
                inv:sendItemAwayMultiple(
                  move,
                  manip.getInventory(),
                  json.data.manipulator,
                  json.data.amount
                )
              end)

              sendResponse(modem, remote.port, hashed, json.id, ok, nil, err)
            elseif json.type == "deposit" then
              local ok, err = pcall(function()
                if json.data.itemName then
                  local move = {}
                  for s, i in pairs(ender_storage.list()) do
                    if i.name == json.data.itemName then
                      table.insert(move, s)
                    end
                  end
                  inv:sendItemAwayMultiple(
                    move,
                    ender_storage,
                    config.remote.ender_storage,
                    json.data.amount
                  )
                elseif json.data.slots then
                  inv:sendItemAwayMultiple(
                    json.data.slots,
                    ender_storage,
                    config.remote.ender_storage,
                    json.data.amount
                  )
                end
              end)

              sendResponse(modem, remote.port, hashed, json.id, ok, nil, err)
            elseif json.type == "list_names" then
              local ok, result = pcall(function()
                return inv:listNames()
              end)
              sendResponse(
                modem,
                remote.port,
                hashed,
                json.id,
                ok,
                ok and result or nil,
                not ok and result or nil
              )
            elseif json.type == "list_item_amounts" then
              local ok, result = pcall(function()
                return inv:listItemAmounts()
              end)

              sendResponse(
                modem,
                remote.port,
                hashed,
                json.id,
                ok,
                ok and result or nil,
                not ok and result or nil
              )
            elseif json.type == "get_item" then
              local ok, result = pcall(function()
                return inv:getItem(json.data.index)
              end)

              sendResponse(
                modem,
                remote.port,
                hashed,
                json.id,
                ok,
                ok and result or nil,
                not ok and result or nil
              )
            end
          end
        end
      end
    end
  end
end

return {
  run = run
}
