local encryption = require("lib.encryption")

---@type string
local password
---@type number
local port

---@type string
local hashedPassword
---@type ccTweaked.peripheral.Modem
local modem
local function sendRequest(reqType, data, wait)
  if wait == nil then
    wait = true
  end
  local id = tostring(math.random(1, 1e9))

  modem.transmit(
    port,
    port,
    encryption.encrypt(hashedPassword,
      textutils.serialiseJSON({
        type = reqType,
        id = id,
        data = data
      })
    )
  )

  if wait then
    -- wait for response
    while true do
      local event, side, senderChannel, replyChannel, message = os.pullEvent("modem_message")
      local ok, decrypted = pcall(encryption.decrypt, hashedPassword, message)
      if not ok then goto continue end

      local resp = textutils.unserialiseJSON(decrypted)
      if resp and resp.type == "response" and resp.id == id then
        return resp.ok, resp.data or resp.error
      end

      ::continue::
    end
  end
end

local function init(modemI, portI, passwordI)
  port = portI
  password = passwordI
  hashedPassword = encryption.hashKey(passwordI)
  modem = modemI

  modem.open(port)
end


---@param wait boolean|nil
local function listNames(wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end
  if wait == nil then
    wait = true
  end

  local ok, data = sendRequest("list_names", nil, wait)
  if wait then
    if not ok then
      error("Failed to list names: " .. tostring(data))
    end
    return data
  end
end

---@param itemName string
---@param amount number|nil
---@param wait boolean|nil
local function withdraw(itemName, amount, wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end

  if wait == nil then
    wait = true
  end

  local ok, data = sendRequest("withdraw", {
    itemName = itemName,
    amount = amount
  }, wait)
  if wait then
    if not ok then
      error("Failed to withdraw: " .. tostring(data))
    end
    return data
  end
end


---@param slots number[]
---@param amount number|nil
---@param wait boolean|nil
local function depositBySlots(slots, amount, wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end

  if wait == nil then
    wait = true
  end

  local data = {
    slots = slots
  }

  if amount then
    data["amount"] = amount
  end

  local ok, response = sendRequest("deposit", data, wait)

  if wait then
    if not ok then
      error("Failed to deposit by slots: " .. tostring(response))
    end
    return response
  end
end

---@param wait boolean|nil
local function listItemAmounts(wait)
  if wait == nil then
    wait = true
  end
  local ok, response = sendRequest("list_item_amounts", nil, wait)

  if wait then
    if not ok then
      error("Failed to list names: " .. tostring(response))
    end
    return response
  end
end


---@param itemName string
---@param wait boolean|nil
local function getItem(itemName, wait)
  if wait == nil then
    wait = true
  end
  local ok, data = sendRequest("get_item", { index = itemName }, wait)
  if wait then
    if not ok then
      error("Failed to list names: " .. tostring(data))
    end
    return data
  end
end


---@param itemName string
---@param amount number|nil
---@param wait boolean|nil
local function depositByItemName(itemName, amount, wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end

  local data = {
    itemName = itemName
  }

  if amount then
    data["amount"] = amount
  end

  local ok, response = sendRequest("deposit", data, wait)

  if wait then
    if not ok then
      error("Failed to list names: " .. tostring(response))
    end
    return response
  end
end

---@param itemName string
---@param manipulator string
---@param amount number|nil
---@param wait boolean|nil
local function depositFromManipulator(manipulator, itemName, amount, wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end

  local data = {
    itemName = itemName,
    manipulator = manipulator
  }

  if amount then
    data["amount"] = amount
  end

  local ok, response = sendRequest("deposit_from_manipulator", data, wait)

  if wait then
    if not ok then
      error("Failed to list names: " .. tostring(response))
    end
    return response
  end
end

---@param itemName string
---@param manipulator string
---@param amount number|nil
---@param wait boolean|nil
local function withdrawToManipulator(manipulator, itemName, amount, wait)
  if not port or not modem or not password then
    error("tiny_ra_library: init was never ran")
  end

  if wait == nil then
    wait = true
  end

  local ok, data = sendRequest("withdraw_to_manipulator", {
    manipulator = manipulator,
    itemName = itemName,
    amount = amount
  }, wait)
  if wait then
    if not ok then
      error("Failed to withdraw: " .. tostring(data))
    end
    return data
  end
end

return {
  init = init,

  listItemAmounts = listItemAmounts,
  getItem = getItem,
  listNames = listNames,

  withdraw = withdraw,
  depositByItemName = depositByItemName,
  depositBySlots = depositBySlots,

  withdrawToManipulator = withdrawToManipulator,
  depositFromManipulator = depositFromManipulator
}
