---@class TurtleItemDetail
---@field name string
---@field count integer
---@field nbt string|nil

---@alias TurtleInventory table<TurtleSlot, ccTweaked.turtle.turtleDetails|ccTweaked.turtle.turtleDetailsDetailed>

---@class InventoryBase
---@field previousInventory TurtleInventory
---@field turtleMoveAllowed boolean
local Base = {}
Base.__index = Base

---Create a new inventory layer instance
---@generic T : InventoryBase
---@param self T
---@return T
function Base.new(self)
  ---@class T
  local o = setmetatable({}, self)

  ---@type TurtleInventory
  o.previousInventory = {}

  ---@type boolean
  o.turtleMoveAllowed = true

  return o
end

---Reads turtle inventory into a slot->item map
---@return TurtleInventory
function Base:getTurtleInventory()
  ---@type TurtleInventory
  local inv = {}

  for slot = 1, 16 do
    inv[slot] = turtle.getItemDetail(slot, false)
  end

  return inv
end

---Automatically detects items inserted by player and exports them
function Base:detectPlayerInsert()
  if not turtle then
    return
  end

  self.previousInventory = self:getTurtleInventory()

  while true do
    os.pullEvent("turtle_inventory")

    local current = self:getTurtleInventory()

    if self.turtleMoveAllowed then
      ---@type TurtleSlot[]
      local newlyAdded = {}

      for slot = 1, 16 do
        if not self.previousInventory[slot] and current[slot] then
          table.insert(newlyAdded, slot)
        end
      end

      if #newlyAdded > 0 then
        ---@diagnostic disable-next-line: param-type-mismatch
        self:sendItemAwayMultiple(newlyAdded)
      end
    end

    self.previousInventory = current
  end
end

---Executes fn while preventing turtle moves
---@generic R
---@param fn fun():R
---@return R
function Base:_withMoveLock(fn)
  self.turtleMoveAllowed = false

  local ok, result = pcall(fn)

  self.turtleMoveAllowed = true

  if ok then
    return result
  end

  error(result)
end

---@param itemName string
---@param perip ccTweaked.peripheral.Inventory|string|nil
---@param maxAmount integer|nil
---@param id string|nil
---@return integer
function Base:sendItemToSelf(itemName, perip, maxAmount, id)
  print("sendItemToSelf not implemented by backend")
  os.exit()
end

---@param slots TurtleSlot[]
---@param perip ccTweaked.peripheral.Inventory|nil
---@param id string|nil
---@param maxAmount integer|nil
---@return integer
function Base:sendItemAwayMultiple(slots, perip, id, maxAmount)
  print("sendItemAwayMultiple not implemented by backend")
  os.exit()
end

---@return string[]
function Base:listNames()
  print("listNames not implemented by backend")
  os.exit()
end

---@return table<string, integer>
function Base:listItemAmounts()
  print("listItemAmounts not implemented by backend")
  os.exit()
end

---@param name string
---@return table
function Base:getItem(name)
  print("getItem not implemented by backend")
  os.exit()
end

---@return fun()
function Base:run()
  print("run not implemented by backend")
  os.exit()
end

function Base:sync()
  print("sync not implemented by backend")
  os.exit()
end

return Base
