---@class InventoryLayerRA : InventoryBase
local Base = require("modules.inventory.base")

---@class InventoryLayerRA : InventoryBase
local RA = setmetatable({}, Base)
RA.__index = RA

local config = require("../../config")

---@alias TurtleSlot integer

local ra = require("tiny_ra_library")

---@class EnderStoragePeripheral : ccTweaked.peripheral.Inventory
---@field list fun():table<TurtleSlot, table>
---@field pushItems fun(to:string|table, fromSlot:integer, amount?:integer, toSlot?:integer):integer
---@field pullItems fun(from:string|table, fromSlot:integer, amount?:integer, toSlot?:integer):integer
---@field setFrequency fun(first: string, second: string, third: string)
---@return InventoryLayerRA
function RA:new()
  ---@class InventoryLayerRA
  local o = Base.new(self)

  ---@type ccTweaked.peripheral.WiredModem|nil
  o.modem = peripheral.wrap(config.remote.modem) --[[@as ccTweaked.peripheral.WiredModem|nil]]

  ---@type EnderStoragePeripheral|nil
  o.ender = peripheral.wrap(config.remote.ender_storage) --[[@as EnderStoragePeripheral?]]

  ---@type string
  ---@diagnostic disable-next-line: undefined-field
  o.turtleId = peripheral.find("modem").getNameLocal()

  return o
end

function RA:sync()
  print("RA already synced")
end

---@return fun()
function RA:run()
  return function()
    ra.init(
      self.modem,
      config.remote.connection.port,
      config.remote.connection.password
    )
  end
end

---@param itemName string
---@param perip ccTweaked.peripheral.Inventory|string|nil
---@param maxAmount integer|nil
---@param id string|nil
---@return integer
function RA:sendItemToSelf(itemName, perip, maxAmount, id)
  if not self.ender then return 0 end

  return self:_withMoveLock(function()
    ra.withdraw(itemName, maxAmount, true)

    ---@type string
    local target =
        (type(perip) == "string" and perip)
        or id
        or self.turtleId

    local total = 0

    for slot in pairs(self.ender.list()) do
      local moved = self.ender.pushItems(target, slot) or 0
      total = total + moved
    end

    return total
  end) or 0
end

---@param slots TurtleSlot[]
---@param perip ccTweaked.peripheral.Inventory|nil
---@param id string|nil
---@param maxAmount integer|nil
---@return integer
function RA:sendItemAwayMultiple(slots, perip, id, maxAmount)
  if not self.ender then return 0 end

  local srcId = id or self.turtleId
  local remaining = maxAmount or math.huge
  local total = 0

  for _, slot in ipairs(slots) do
    if remaining <= 0 then break end

    local pulled =
        self.ender.pullItems(srcId, slot, math.min(64, remaining)) or 0

    remaining = remaining - pulled
    total = total + pulled
  end

  ra.depositBySlots(self.ender.list(), nil, false)

  return total
end

---@return string[]
function RA:listNames()
  return ra.listNames()
end

---@return table<string, integer>
function RA:listItemAmounts()
  return ra.listItemAmounts()
end

---@param name string
---@return table
function RA:getItem(name)
  return ra.getItem(name)
end

---@return InventoryLayerRA
return function()
  return RA:new()
end
